' sap-script-complete-tool-workflow.vbs - Complete tool workflow: Plant check + Tool extraction + All dates/intervals
' Combined script that handles the entire tool processing workflow in a single ZLIFECYCLE transaction
' Includes: Plant/SLOC routing logic, Tool information extraction, and All dates/intervals extraction (Calibration, Certification, Control, NEN3140)
On Error Resume Next

' Set scripting mode to true
Set WSHShell = CreateObject("WScript.Shell")

' Disable SAP GUI scripting warnings in registry
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' Declare log file object and path
Dim logFile, logFilePath

' Get Downloads folder path
Set WShell = CreateObject("WScript.Shell")
DownloadsPath = WShell.ExpandEnvironmentStrings("%USERPROFILE%") & "\Downloads\"

' Get barcode ID from command line argument
Dim barcodeId
barcodeId = WScript.Arguments.Item(0)

' Initialize log file if path is provided
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    ' Try to open the log file in append mode (8)
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Log script start
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Barcode ID: " & barcodeId
        LogToFile "========================================="
    End If
End If

' SAP GUI connection
If Not IsObject(application) Then
    Set SapGuiAuto = GetObject("SAPGUI")
    Set application = SapGuiAuto.GetScriptingEngine
End If
If Not IsObject(connection) Then
    Set connection = application.Children(0)
End If
If Not IsObject(session) Then
    Set session = connection.Children(0)
End If
If IsObject(WScript) Then
    WScript.ConnectObject session, "on"
    WScript.ConnectObject application, "on"
End If

' Handle security dialog if needed
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    Do While timeout < 10
        If shell.AppActivate("SAP GUI Security") Then
            WScript.Sleep 500
            shell.SendKeys " "  ' Press space to check "Remember my decision"
            WScript.Sleep 200
            shell.SendKeys "{ENTER}"  ' Press Enter to click Allow
            dialogFound = True
            Exit Do
        End If
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' Call security dialog handler
HandleSecurityDialog()

' Launch transaction
session.findById("wnd[0]").maximize
session.findById("wnd[0]/tbar[0]/okcd").text = "zlifecycle"
session.findById("wnd[0]").sendVKey 0
WScript.Sleep 2000

' Enter equipment number
session.findById("wnd[0]/usr/txtP_BARC").text = barcodeId
session.findById("wnd[0]/usr/txtP_BARC").caretPosition = Len(barcodeId)
session.findById("wnd[0]").sendVKey 8
WScript.Sleep 2000

' ============================================================
' PART 1: PLANT CHECK LOGIC (from sap-script-plant-check.vbs)
' ============================================================

' Variables for plant check
Dim grid, rowCount, rowIndex, recPlant, slocTo
Set grid = session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell")

recPlant = ""
slocTo = ""

If Not grid Is Nothing Then
    rowCount = grid.RowCount

    For rowIndex = 0 To rowCount - 1
        recPlant = Trim(grid.GetCellValue(rowIndex, "UMWRK"))
        slocTo   = Trim(grid.GetCellValue(rowIndex, "UMLGO"))

        If recPlant <> "" And slocTo <> "" Then
            Exit For
        End If
    Next
End If

' Output plant check results in a parsable format
LogMessage "PLANT_CHECK:RecPlant:" & recPlant
LogMessage "PLANT_CHECK:SlocTo:" & slocTo

' Determine routing logic result
Dim routingResult
If (recPlant = "NL10" And slocTo = "D008") Then
    routingResult = "CONTINUE_NORMAL"
ElseIf (recPlant = "NL10" And slocTo <> "D008") Then
    routingResult = "GO_TO_TEAM_LEADER"
ElseIf (recPlant = "NL05" And slocTo = "D008") Then
    routingResult = "USE_TMM_WEBFORM"
ElseIf (recPlant = "NL05" And slocTo <> "D008") Then
    routingResult = "GO_TO_TEAM_LEADER"
Else
    routingResult = "GO_TO_TEAM_LEADER"
End If

LogMessage "ROUTING_RESULT:" & routingResult

' ============================================================
' ROUTING DECISION: Only continue with main script if CONTINUE_NORMAL
' ============================================================

If routingResult <> "CONTINUE_NORMAL" Then
    ' Return to SAP main screen and exit
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    
    ' Close log file and exit
    CloseLogFile()
    WScript.Quit
End If

' ============================================================
' PART 2: MAIN SCRIPT LOGIC (from sap-script.vbs)
' Continue in the same transaction since routing is CONTINUE_NORMAL
' ============================================================

' Select the first row in the grid (we're already in ZLIFECYCLE transaction)
session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").currentCellColumn = "SERNR"
session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").selectedRows = "0"
session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").clickCurrentCell

' Check if it's a stepper part (type U)
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-EQTYP").setFocus
equipmentType = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-EQTYP").text

' Check if it's a stepper part (type U)
If equipmentType = "U" Then
    ' Return a special error code to indicate stepper part
    LogMessage "ERROR:STEPPER_PART"
    
    ' Go back to previous screens so user returns to ZLIFECYCLE search screen
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    session.findById("wnd[0]/tbar[0]/btn[3]").press
    
    ' Close log file and exit
    CloseLogFile()
    WScript.Quit
End If

' Get the values from the grid directly
On Error Resume Next
equipmentID = session.findById("wnd[0]/usr/cntlGRID1/shellcont/shell/shellcont[1]/shell").getCellValue(0, "EQUNR")
On Error GoTo 0

' Here you get the description - use .text to get the value
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152B:SAPLITO0:1530/txtITOB-SHTXT").setFocus
description = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152B:SAPLITO0:1530/txtITOB-SHTXT").text

' Here you get the Material Number
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-MATNR").setFocus
materialNumber = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/ctxtITOB-MATNR").text

' Use the correct field for Serial Number
session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/txtITOB-SERNR").setFocus
serialNumber = session.findById("wnd[0]/usr/subSUB_EQKO:SAPLITO0:0152/subSUB_0152A:SAPLITO0:1521/txtITOB-SERNR").text

' Navigate to tab 06 for Equipment Number
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\06").select
WScript.Sleep 500

' Here you get the Equipment Number
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\06/ssubSUB_DATA:SAPLITO0:0122/subSUB_0122A:SAPLITO0:1520/ctxtITOBATTR-EQUNR").setFocus
equipmentNumber = session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\06/ssubSUB_DATA:SAPLITO0:0122/subSUB_0122A:SAPLITO0:1520/ctxtITOBATTR-EQUNR").text

' Navigate to tab 07 for Barcode ID
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\07").select

' Here you get the Barcode ID
session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\07/ssubSUB_DATA:SAPLITO0:0102/subSUB_0102B:SAPLITO0:1080/ssubXUSR1080:SAPLXTOB:1003/txtITOB-ZZBARCODE").setFocus
barcodeIdFromSAP = session.findById("wnd[0]/usr/tabsTABSTRIP/tabpT\07/ssubSUB_DATA:SAPLITO0:0102/subSUB_0102B:SAPLITO0:1080/ssubXUSR1080:SAPLXTOB:1003/txtITOB-ZZBARCODE").text

' ============================================================
' PART 3: DIRECT DATES EXTRACTION (from sap-script-direct-dates.vbs)
' Extract all dates and intervals while staying in the same ZLIFECYCLE transaction
' Includes: Calibration, Certification, Control, and NEN3140 dates/intervals
' ============================================================

' Fields to look up in the SAP grid and their display names
Dim labels, values, dates_i, dates_f
labels = Array( _
    Array("Next Calibration Date", "Calibration Date"), _
    Array("Next Certification Date", "Certification Date"), _
    Array("Next Control Date", "Control Date"), _
    Array("Next NEN3140 Control Date", "NEN3140 Control Date"), _
    Array("Calibration Interval", "Calibration Interval"), _
    Array("Certification Interval", "Certification Interval"), _
    Array("Control/Inspection Interval", "Control/Inspection Interval"), _
    Array("NEN3140 Control Interval", "NEN3140 Control Interval") _
)

' Open Class Overview (staying in ZLIFECYCLE context)
LogMessage "Opening Class Overview..."
session.FindById("wnd[0]/tbar[1]/btn[20]").Press

' Check if classification tab is available
On Error Resume Next
Dim classificationAvailable
classificationAvailable = True
session.FindById("wnd[0]/usr/subSUBSCR_BEWERT:SAPLCTMS:5000/tabsTABSTRIP_CHAR/tabpTAB1/ssubTABSTRIP_CHAR_GR:SAPLCTMS:5100/tblSAPLCTMSCHARS_S")
If Err.Number <> 0 Then
    classificationAvailable = False
    LogMessage "WARNING: Classification data not available for this equipment"
End If
On Error GoTo 0

If classificationAvailable Then
    ' Initialize results dictionary
    Set values = CreateObject("Scripting.Dictionary")    ' Loop through each field we want to extract
    For dates_f = 0 To UBound(labels)
        Dim fieldLabel, displayName, dates_grid, foundValue
        fieldLabel = labels(dates_f)(0)
        displayName = labels(dates_f)(1)
        
        LogMessage "Searching for field: " & fieldLabel

        ' Use Position button to scroll to field
        On Error Resume Next
        session.FindById("wnd[0]/usr/subSUBSCR_BEWERT:SAPLCTMS:5000/tabsTABSTRIP_CHAR/tabpTAB1/ssubTABSTRIP_CHAR_GR:SAPLCTMS:5100/btnRCTMS-AUFS").Press
        
        If Err.Number <> 0 Then
            LogMessage "WARNING: Could not access Position button for field: " & fieldLabel
            foundValue = "Not Found"
            values.Add displayName, foundValue
            Err.Clear
        Else
            On Error GoTo 0
            
            session.FindById("wnd[1]/usr/txtCLHP-CR_STATUS_TEXT").Text = fieldLabel
            session.FindById("wnd[1]").SendVKey 0
            ' Get grid and scan for value
            On Error Resume Next
            Set dates_grid = session.FindById("wnd[0]/usr/subSUBSCR_BEWERT:SAPLCTMS:5000/tabsTABSTRIP_CHAR/tabpTAB1/ssubTABSTRIP_CHAR_GR:SAPLCTMS:5100/tblSAPLCTMSCHARS_S")
            
            If Err.Number <> 0 Then
                LogMessage "WARNING: Could not access grid for field: " & fieldLabel
                foundValue = "Not Found"
                values.Add displayName, foundValue
                Err.Clear
            Else
                On Error GoTo 0
                  foundValue = ""
                ' Scan grid for the field
                For dates_i = 0 To dates_grid.Rows.Count - 1
                    On Error Resume Next
                    Dim cellValue
                    cellValue = dates_grid.GetCell(dates_i, 0).Text
                    If Err.Number = 0 And cellValue = fieldLabel Then
                        foundValue = dates_grid.GetCell(dates_i, 1).Text
                        If Err.Number = 0 Then
                            If foundValue = "" Then
                                LogMessage "Found " & fieldLabel & " but value is empty"
                                foundValue = "N/A"
                            Else
                                LogMessage "Found " & fieldLabel & ": " & foundValue
                            End If
                            Exit For
                        End If
                    End If
                    Err.Clear
                    On Error GoTo 0
                Next

                If foundValue = "" Then
                    foundValue = "Not Found"
                    LogMessage "Field not found in grid: " & fieldLabel
                End If

                values.Add displayName, foundValue
            End If
        End If
    Next
    
    ' Output all dates and intervals fields in a structured format
    LogMessage "Building dates and intervals output results..."
    For Each displayName In values.Keys
        ' Format: DATE_FIELD:DisplayName:Value
        LogMessage "DATE_FIELD:" & displayName & ":" & values(displayName)
    Next
Else
    ' Output that no dates and intervals data is available
    LogMessage "DATE_DATA_INFO:No classification data available"
End If

' Return to previous screens
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press
session.findById("wnd[0]/tbar[0]/btn[3]").press

' Echo the results in a format parsable by our JavaScript
LogMessage "SAP_FIELD:Description:" & description
LogMessage "SAP_FIELD:SerialNumber:" & serialNumber
LogMessage "SAP_FIELD:EquipmentNumber:" & equipmentNumber
LogMessage "SAP_FIELD:BarcodeID:" & barcodeIdFromSAP
LogMessage "SAP_FIELD:MaterialNumber:" & materialNumber
LogMessage "SAP_FIELD:EquipmentType:" & equipmentType

' ============================================================
' HELPER FUNCTIONS
' ============================================================

' Function to log message to file
Sub LogToFile(message)
    On Error Resume Next
    If Not logFile Is Nothing Then
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

' Function to log message to both console and file
Sub LogMessage(message)
    WScript.Echo message
    LogToFile message
End Sub

' Add at the end of your script before it exits
Sub CloseLogFile()
    On Error Resume Next
    If Not logFile Is Nothing Then
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        logFile.Close
        Set logFile = Nothing
    End If
End Sub

' At the end of the script, add:
CloseLogFile()
